<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/rbac.php';
require_once '../includes/functions.php';
require_once '../includes/middleware.php';

// Check if user is admin
adminOnly();

$user = getUserContext();
$pdo = getDB();

$page_title = 'Role Management';
include 'includes/admin_header.php';
?>

<div class="roles-management">
    <div class="management-header">
        <h1><i class="fas fa-shield-alt"></i> Role Management</h1>
        <p>View and understand user roles and their permissions in the system.</p>
    </div>

    <div class="roles-overview">
        <div class="roles-section">
            <h2>Role Hierarchy</h2>
            <div class="role-hierarchy">
                <?php foreach (ROLES as $role => $level): ?>
                    <div class="hierarchy-item">
                        <div class="role-badge role-<?php echo $role; ?> large">
                            <?php echo getRoleDisplayName($role); ?>
                        </div>
                        <div class="role-level">Level <?php echo $level; ?></div>
                        <div class="role-description">
                            <?php
                            switch ($role) {
                                case 'guest':
                                    echo 'No account, limited access';
                                    break;
                                case 'student':
                                    echo 'Basic user access, can view content';
                                    break;
                                case 'instructor':
                                    echo 'Can create and manage content';
                                    break;
                                case 'admin':
                                    echo 'Full system access and user management';
                                    break;
                            }
                            ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <div class="permissions-section">
            <h2>Permissions Matrix</h2>
            <div class="permissions-table">
                <table>
                    <thead>
                        <tr>
                            <th>Permission</th>
                            <?php foreach (array_keys(ROLES) as $role): ?>
                                <th><?php echo getRoleDisplayName($role); ?></th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (PERMISSIONS as $permission => $allowed_roles): ?>
                            <tr>
                                <td class="permission-name">
                                    <?php
                                    $permission_labels = [
                                        'view_posts' => 'View Posts',
                                        'create_posts' => 'Create Posts',
                                        'edit_own_posts' => 'Edit Own Posts',
                                        'edit_all_posts' => 'Edit All Posts',
                                        'delete_posts' => 'Delete Posts',
                                        'manage_categories' => 'Manage Categories',
                                        'manage_users' => 'Manage Users',
                                        'manage_site_settings' => 'Manage Site Settings',
                                        'manage_announcements' => 'Manage Announcements',
                                        'view_admin_dashboard' => 'View Admin Dashboard',
                                        'manage_students' => 'Manage Students',
                                        'view_analytics' => 'View Analytics'
                                    ];
                                    echo $permission_labels[$permission] ?? ucwords(str_replace('_', ' ', $permission));
                                    ?>
                                </td>
                                <?php foreach (array_keys(ROLES) as $role): ?>
                                    <td class="permission-status">
                                        <?php if (in_array($role, $allowed_roles)): ?>
                                            <i class="fas fa-check text-success"></i>
                                        <?php else: ?>
                                            <i class="fas fa-times text-muted"></i>
                                        <?php endif; ?>
                                    </td>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="users-by-role-section">
            <h2>Users by Role</h2>
            <div class="role-stats">
                <?php
                $role_counts = $pdo->query("SELECT role, COUNT(*) as count FROM users GROUP BY role ORDER BY FIELD(role, 'admin', 'instructor', 'student', 'guest')")->fetchAll(PDO::FETCH_KEY_PAIR);
                $total_users = array_sum($role_counts);
                ?>

                <?php foreach ($role_counts as $role => $count): ?>
                    <div class="role-stat-card">
                        <div class="role-stat-header">
                            <span class="role-name"><?php echo getRoleDisplayName($role); ?></span>
                            <span class="role-count"><?php echo $count; ?> users</span>
                        </div>
                        <div class="role-stat-bar">
                            <div class="role-stat-fill" style="width: <?php echo $total_users > 0 ? ($count / $total_users * 100) : 0; ?>%"></div>
                        </div>
                        <div class="role-percentage"><?php echo $total_users > 0 ? round(($count / $total_users * 100), 1) : 0; ?>%</div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <div class="role-management-info">
            <h2>Role Management Notes</h2>
            <div class="info-cards">
                <div class="info-card">
                    <h3><i class="fas fa-info-circle"></i> How Roles Work</h3>
                    <p>Roles determine what actions users can perform in the system. Higher-level roles inherit permissions from lower-level roles.</p>
                </div>

                <div class="info-card">
                    <h3><i class="fas fa-user-cog"></i> Changing User Roles</h3>
                    <p>Administrators can change user roles through the <a href="users.php">User Management</a> interface. Be careful when assigning admin privileges.</p>
                </div>

                <div class="info-card">
                    <h3><i class="fas fa-lock"></i> Permission System</h3>
                    <p>Permissions are checked throughout the application. The system uses role-based access control (RBAC) for security.</p>
                </div>

                <div class="info-card">
                    <h3><i class="fas fa-exclamation-triangle"></i> Security Considerations</h3>
                    <p>Only grant the minimum necessary permissions. Regularly review user roles and remove unnecessary admin access.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.roles-management {
    max-width: 1200px;
    margin: 0 auto;
}

.management-header {
    margin-bottom: 2rem;
}

.management-header h1 {
    color: #333;
    margin-bottom: 0.5rem;
}

.management-header p {
    color: #666;
    font-size: 1.1rem;
}

.role-hierarchy {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.hierarchy-item {
    background: #f8f9fa;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 1.5rem;
    text-align: center;
}

.role-badge.large {
    font-size: 1.2rem;
    padding: 0.5rem 1rem;
    margin-bottom: 0.5rem;
}

.role-level {
    font-weight: bold;
    color: #666;
    margin-bottom: 0.5rem;
}

.role-description {
    color: #666;
    font-size: 0.9rem;
}

.permissions-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    margin-bottom: 2rem;
}

.permissions-table table {
    width: 100%;
    border-collapse: collapse;
}

.permissions-table th,
.permissions-table td {
    padding: 0.75rem;
    text-align: center;
    border-bottom: 1px solid #e9ecef;
}

.permissions-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #333;
}

.permission-name {
    text-align: left !important;
    font-weight: 500;
}

.permission-status {
    font-size: 1.2rem;
}

.text-success {
    color: #28a745;
}

.text-muted {
    color: #6c757d;
}

.role-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.role-stat-card {
    background: white;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 1.5rem;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.role-stat-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.role-name {
    font-weight: 600;
    color: #333;
}

.role-count {
    color: #666;
    font-size: 0.9rem;
}

.role-stat-bar {
    height: 8px;
    background: #e9ecef;
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.role-stat-fill {
    height: 100%;
    background: linear-gradient(90deg, #007bff, #0056b3);
    border-radius: 4px;
    transition: width 0.3s ease;
}

.role-percentage {
    text-align: center;
    font-weight: 600;
    color: #333;
    font-size: 0.9rem;
}

.info-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.info-card {
    background: white;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 1.5rem;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.info-card h3 {
    color: #333;
    margin-bottom: 0.5rem;
    font-size: 1rem;
}

.info-card p {
    color: #666;
    font-size: 0.9rem;
    line-height: 1.4;
}

.info-card a {
    color: #007bff;
    text-decoration: none;
}

.info-card a:hover {
    text-decoration: underline;
}
</style>

<?php include 'includes/admin_footer.php'; ?>